/*
 * Copyright 2022 Devin Lin <devin@kde.org>
 * SPDX-License-Identifier: LGPL-2.0-or-later
 */

import QtQuick
import QtQuick.Controls as Controls
import QtQuick.Layouts

import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as MobileForm
import org.kde.kirigamiaddons.delegates as Delegates
import org.kde.kirigamiaddons.components as Components

Kirigami.ApplicationWindow {
    id: appwindow

    title: "Kirigami Addons Delegates Test"

    width: Kirigami.Settings.isMobile ? 400 : 800
    height: Kirigami.Settings.isMobile ? 550 : 500

    pageStack {
        defaultColumnWidth: Kirigami.Units.gridUnit * 35
        initialPage: welcomePageComponent

        globalToolBar {
            style: Kirigami.ApplicationHeaderStyle.ToolBar;
            showNavigationButtons: Kirigami.ApplicationHeaderStyle.ShowBackButton;
        }
    }


    // Dummy implementation of ki18n
    function i18nd(context, text) {
        return text;
    }

    function i18ndp(context, text1, text2, number) {
        return number === 1 ? text1 : text2;
    }

    function i18ndc(context, text) {
        return text
    }

    function i18nc(context, text) {
        return text;
    }

    LayoutMirroring.enabled: false

    Component {
        id: roundedItemDelegatePageComponent

        Kirigami.ScrollablePage {
            title: "RoundedItemDelegate"

            header: Components.Banner {
                text: "Hello"
                visible: true
            }

            ListView {
                model: 50
                delegate: Delegates.RoundedItemDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    icon.name: "kde"
                    text: "Item " + modelData
                }

                Components.DoubleFloatingButton {
                    anchors {
                        right: parent.right
                        rightMargin: Kirigami.Units.largeSpacing
                        bottom: parent.bottom
                        bottomMargin: Kirigami.Units.largeSpacing
                    }

                    leadingAction: Kirigami.Action {
                        text: "Hello"
                        icon.name: "list-add"
                    }

                    trailingAction: Kirigami.Action {
                        text: "Hello"
                        icon.name: "list-add"
                    }
                }
            }
        }
    }

    Component {
        id: subtitleRoundedItemDelegatePageComponent

        Kirigami.ScrollablePage {
            title: "RoundedItemDelegate with subtitle"

            header: Components.Banner {
                title: "Hello"
                text: "Hello world"
                visible: true
            }

            ListView {
                model: 50

                delegate: Delegates.RoundedItemDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    icon.name: "kde"
                    text: "Item " + modelData

                    contentItem: Delegates.SubtitleContentItem {
                        itemDelegate: delegate
                        subtitle: "Subtitle " + delegate.modelData
                    }
                }

                Components.FloatingButton {
                    anchors {
                        right: parent.right
                        rightMargin: Kirigami.Units.largeSpacing
                        bottom: parent.bottom
                        bottomMargin: Kirigami.Units.largeSpacing
                    }

                    action: Kirigami.Action {
                        text: "Hello"
                        icon.name: "list-add"
                    }
                }
            }
        }
    }

    Component {
        id: indicatorItemDelegatePageComponent

        Kirigami.ScrollablePage {
            title: "IndicatorItemDelegate"

            ListView {
                id: listView

                model: 50
                delegate: Delegates.IndicatorItemDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    unread: Math.random() > 0.3
                    icon.name: "kde"
                    text: "Item " + modelData
                    onClicked: {
                        unread = false;
                        listView.currentIndex = index;
                    }
                }
            }
        }
    }

    Component {
        id: subtitleIndicatorItemDelegatePageComponent

        Kirigami.ScrollablePage {
            title: "IndicatorItemDelegate with subtitle"

            ListView {
                id: listView

                model: 50
                delegate: Delegates.IndicatorItemDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    unread: Math.random() > 0.3
                    icon.name: "kde"
                    text: "Item " + modelData
                    onClicked: {
                        unread = false;
                        listView.currentIndex = index;
                    }

                    contentItem: Delegates.SubtitleContentItem {
                        itemDelegate: delegate
                        subtitle: "Subtitle " + delegate.modelData
                    }
                }
            }
        }
    }

    Component {
        id: checkDelegateComponent

        Kirigami.ScrollablePage {
            title: "CheckDelegate"

            ListView {
                id: listView

                currentIndex: -1
                model: 50
                delegate: Delegates.CheckDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    text: "Item " + modelData
                    onClicked: {
                        listView.currentIndex = index;
                    }
                }
            }
        }
    }

    Component {
        id: checkDelegateSubtitleComponent

        Kirigami.ScrollablePage {
            title: "CheckDelegate with subtitle"

            ListView {
                id: listView

                currentIndex: -1
                model: 50
                delegate: Delegates.CheckDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    text: "Item " + modelData
                    onToggled: {
                        console.log("Item was toggled", modelData, checked)
                    }

                    contentItem: Delegates.SubtitleContentItem {
                        itemDelegate: delegate
                        subtitle: "Subtitle " + delegate.modelData
                    }
                }
            }
        }
    }

    Component {
        id: radioDelegateComponent

        Kirigami.ScrollablePage {
            title: "RadioDelegate"

            ListView {
                id: listView

                currentIndex: -1
                model: 50
                delegate: Delegates.RadioDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    text: "Item " + modelData
                    onClicked: {
                        listView.currentIndex = index;
                    }
                }
            }
        }
    }

    Component {
        id: radioDelegateSubtitleComponent

        Kirigami.ScrollablePage {
            title: "RadioDelegate with subtitle"

            ListView {
                id: listView

                currentIndex: -1
                model: 50
                delegate: Delegates.RadioDelegate {
                    id: delegate

                    required property int index
                    required property int modelData

                    text: "Item " + modelData
                    onToggled: {
                        console.log("Item was toggled", modelData, checked)
                    }

                    contentItem: Delegates.SubtitleContentItem {
                        itemDelegate: delegate
                        subtitle: "Subtitle " + delegate.modelData
                    }
                }
            }
        }
    }

    Component {
        id: welcomePageComponent

        Kirigami.ScrollablePage {
            id: page
            title: "Mobile Form Layout"

            leftPadding: 0
            rightPadding: 0
            topPadding: Kirigami.Units.gridUnit
            bottomPadding: Kirigami.Units.gridUnit

            ColumnLayout {
                spacing: 0

                // Form Grid
                MobileForm.FormGridContainer {
                    id: container

                    Layout.fillWidth: true

                    infoCards: [
                        MobileForm.FormGridContainer.InfoCard {
                            title: "RoundedItemDelegate"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(roundedItemDelegatePageComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "ReadIndicatorItemDelegate"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(indicatorItemDelegatePageComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "RoundedItemDelegate with subtitle"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(subtitleRoundedItemDelegatePageComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "ReadIndicatorItemDelegate with subtitle"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(subtitleIndicatorItemDelegatePageComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "CheckDelegate"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(checkDelegateComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "CheckDelegate with subtitle"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(checkDelegateSubtitleComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "RadioDelegate"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(radioDelegateComponent);
                            }
                        },
                        MobileForm.FormGridContainer.InfoCard {
                            title: "RadiokDelegate with subtitle"
                            action: Kirigami.Action {
                                onTriggered: applicationWindow().pageStack.push(radioDelegateSubtitleComponent);
                            }
                        }
                    ]
                }
            }
        }
    }
}
